/*
  Settings screen for Bombing Run.
  
  Author Mark Caldwell aka W@rHe@d of The Reliquary
  
  General game settings can be maintained here rather than by editing ini file.
  Settings can be per game and per map.
  User can select which mutators they want to run with game.
  Access from client pc Online available to admins. 
*/

class UTBRGameSettingsScene extends UTUIFrontEnd config(BombingRun);

var UTBRSettings Settings;

var UIObject Button_Save, Button_Cancel, Button_Delete, Button_Defaults, Button_Restart;
var UTUIComboBox mapComboBox;
var UTUIComboBox mutatorComboBox;
var UTUIDataStore_StringList StringStore;
var bool disableEvents;
var UILabel Label_ToolTip, Label_ToolTip2;
var bool inGame;
var Array<string> MapList;
var Array<string> MutatorNameList;
var Array<string> MutatorClassList;
var UTBRPlayerManager PlayerManager;


event PostInitialize()
{ 
  super.PostInitialize();

  //editor does not call PostInitialize 
  inGame = true;
}

function UTBRSettings LoadSettings(string map)
{
    local UTBRSettings ret;
       
    if (map == "")
    {
        map = "Bombing Run Settings";
    }
    
    ret = new (none, map) class'BombingRun.UTBRSettings';
    return ret;  
}

event SceneActivated( bool bInitialActivation )
{
    super.SceneActivated(bInitialActivation);

    if (inGame)
    {
        Init();
    }
}

function ProcessValue(string which, out string value, bool saving)
{
    local UIObject uiobject;
    local float fValue;
    
   
    uiobject = FindChild(name("Field_" $ which), true);
    
    if (! saving)
    {
        uiobject.OnQueryToolTip = OnQueryToolTip;
    }
    
    if (UTUISlider(uiobject) != none)
    {
        if (! saving)
        {
            fvalue = float(value);
            
            if (which == "GameSpeedMultiplier")
            {
                fvalue = fvalue * 100;
            }

            UTUISlider(uiobject).SetValue(fvalue);
            UTUISlider(uiobject).UpdateCaption();
        }
        else
        {
            fvalue = UTUISlider(uiobject).GetValue();
                    
            if (which == "GameSpeedMultiplier")
            {
                fvalue = fvalue / 100;
            }
            
            value = string(fvalue);
        }
    }
    
    if (UICheckBox(uiobject) != none)
    {
        if (! saving)
        {
            UICheckBox(uiobject).SetValue(bool(value));
        }
        else
        {
            value = string(UICheckBox(uiobject).IsChecked());
        }
    }    
}

//return true if accessing from remote station Online
function bool IsRemote()
{
    return (UTBRGame(GetUTPlayerOwner().WorldInfo.Game) != none) && (GetUTPlayerOwner().WorldInfo.NetMode != NM_Standalone);
}
    
function Init()
{
    local UITabControl tc;
    
    if (IsRemote())
    {
        //remote Online admin access
        PlayerManager = UTBRGame(GetUTPlayerOwner().WorldInfo.Game).LocalPlayerManager;      
    } 
    else  if (UTBRGame(GetUTPlayerOwner().WorldInfo.Game) != none)
    {
        //Instant Action
        PlayerManager = UTBRGame(GetUTPlayerOwner().WorldInfo.Game).GetPlayerManager(GetUTPlayerOwner().PlayerReplicationInfo);
    }
    else
    {
        //UT main menu
        PlayerManager = GetUTPlayerOwner().Spawn(class'UTBRPlayerManager');
        PlayerManager.Controller = GetUTPlayerOwner();
    }
                        
    PlayerManager.SettingsScreen = self;    

    if ((PlayerManager == none) || (! PlayerManager.AllowAdminAccess()))
    {
        CloseScene(self);
        return;
    }
              
    tc = UITabControl(FindChild('UTUITabControl_TabControl', true));
    tc.OnPageActivated = OnPageActivated;
    
    Label_ToolTip = UILabel(FindChild('Field_Tooltip', true));
    Label_ToolTip2 = UILabel(FindChild('Field_Tooltip2', true));    
    	     
    Button_Save = FindChild('Button_Save', true);
    Button_Save.OnClicked = OnClicked;

    Button_Cancel = FindChild('Button_Cancel', true);
    Button_Cancel.OnClicked = OnClicked;
    
    Button_Delete = FindChild('Button_Delete', true);
    Button_Delete.OnClicked = OnClicked;    
    
    Button_Defaults = FindChild('Button_Defaults', true);
    Button_Defaults.OnClicked = OnClicked;
    
    Button_Restart = FindChild('Button_Restart', true);
    Button_Restart.OnClicked = OnClicked;
    Button_Restart.OnQueryToolTip = OnQueryToolTip;          
       
    StringStore = UTUIDataStore_StringList (ResolveDataStore ('UTStringList'));
     
     
    mapComboBox = UTUIComboBox(FindChild('Field_Maps', true));   
    mapComboBox.OnValueChanged = MapChanged;
  
    mutatorComboBox = UTUIComboBox(FindChild('Field_Mutators', true));
    mutatorComboBox.OnValueChanged = MutatorChanged;

    
    MapList.Length = 0;
    MutatorNameList.Length = 0;
    MutatorClassList.Length = 0;

    //alert all someone has gone into settings Online
    PlayerManager.SendSettingsScreenAlert();
    
    //update main fields first and then load up maps and mutators so remote user can start working if maps and muts take long to get  
    PlayerManager.LoadSettingsScreenSettings("");
    
    PlayerManager.LoadSettingsScreenMaps();
    PlayerManager.LoadSettingsScreenMutators();
}

function SetFields()
{
    DisableEvents = true;          
    ProcessSettings(false);           
    DisableEvents = false;           
}

function GetMapList()
{
	  local UTMapList MLObj;    
    local string changed;  
    local UTBRSettings mapSettings;
    local int i;
                
    MapList.Length = 0;
   
 		MLObj = new(none) Class'UTMapList';
 		MLObj.AutoLoadPrefixes = "BR";
		MLObj.Initialize();
    for (i = 0; i < MLObj.Maps.Length; i++)
    {
        changed = "";
        mapSettings = LoadSettings(MLObj.Maps[i].Map);
        if (mapSettings.BallSpeed != "")
        {
            //map settings exist
            changed = ">";
        }
        
        MapList[MapList.Length] = changed $ MLObj.Maps[i].Map;
    }
}

function SetMaps()
{
    local int i;
    local int mapBoxRowCount;
     
    if (mapComboBox == none)
    {
        return;
    }
    
    disableEvents = true;
              
    StringStore.Empty ('BR_MAP_COMBO');    
    StringStore.AddStr('BR_MAP_COMBO', "None");
    
    mapBoxRowCount = 0;
   
    for (i = 0; i < MapList.Length; i++)
    {
        StringStore.AddStr('BR_MAP_COMBO', MapList[i]);
        mapBoxRowCount++;
    }            
       
    mapComboBox.ComboList.SetDataStoreBinding("<UTStringList:BR_MAP_COMBO>");
    mapComboBox.ComboList.RefreshSubscriberValue();
    
    if (mapBoxRowCount > 10) 
    {
        mapBoxRowCount = 10;
    }
   
    mapComboBox.ComboList.SetRowCount(mapBoxRowCount);
    
    disableEvents = false;         
}

function MapChanged( UIObject Sender, int PlayerIndex )
{   
    if (disableEvents)
    {
        return;
    }
    
    if (PlayerManager != none)
    {
        PlayerManager.LoadSettingsScreenSettings(CurrentMap());
    }
}

function MutatorChanged( UIObject Sender, int PlayerIndex )
{
    local string mutator;
    
    if (disableEvents)
    {
        return;
    }
    
		mutator = mutatorComboBox.ComboList.GetElementValue(mutatorComboBox.ComboList.Index);
    if (mid(mutator, 0, 1) == ">")
    {
        mutator = mid(mutator, 1);
    }
    
    MarkCombo("TOGGLE", mutatorComboBox);		
}

function GetMutatorList()
{
    local int i, j;
    local UTUIDataStore_MenuItems MenuDataStore;
    local array<UTUIResourceDataProvider> mutProviders;

    MutatorNameList.Length = 0;
    MutatorClassList.Length = 0;
            
    MenuDataStore = UTUIDataStore_MenuItems(ResolveDataStore('UTMenuItems'));
    menuDataStore.GetAllResourceDataProviders(class'UTUIDataProvider_Mutator', mutProviders);
    
    for (i = 0; i < mutProviders.Length; i++)
    {   
        for (j = 0; j <= MutatorNameList.Length; j++)
        {
            if ((j == MutatorNameList.Length) || (UTUIDataProvider_Mutator(mutProviders[i]).FriendlyName < MutatorNameList[j]))
            {
                MutatorNameList.Insert(j, 1);
                MutatorClassList.Insert(j, 1);
                                
                MutatorNameList[j] = UTUIDataProvider_Mutator(mutProviders[i]).FriendlyName;
                MutatorClassList[j] = UTUIDataProvider_Mutator(mutProviders[i]).ClassName;
                
                break;
            }
        }
    }
}

function SetMutators()
{
    local int i;
    local int mutatorBoxRowCount;
    local string used;
    local int selectedRow;
    
    if (mutatorComboBox == none)
    {
        return;
    }

    disableEvents = true;  
    
    StringStore.Empty ('BR_MUTATOR_COMBO');    
    
    mutatorBoxRowCount = 0;
    selectedRow = -1;
        
    for (i = 0; i < MutatorNameList.Length; i++)
    {
        used = "";
        if (instr(Settings.Mutators, MutatorClassList[i]) >= 0)
        {
            used = ">";
            if (selectedRow == -1)
            {
                selectedRow = i;
            }
        }
        
        StringStore.AddStr('BR_MUTATOR_COMBO', used $ MutatorNameList[i]);
               
        mutatorBoxRowCount++;       
    }
    
    if (selectedRow == -1)
    {
        selectedRow = 0;
    }

 
    mutatorComboBox.ComboList.SetDataStoreBinding("<UTStringList:BR_MUTATOR_COMBO>");
    mutatorComboBox.ComboList.RefreshSubscriberValue();
    
    if (mutatorBoxRowCount > 8) 
    {
        mutatorBoxRowCount = 8;
    }
   
    mutatorComboBox.ComboList.SetRowCount(mutatorBoxRowCount);
    mutatorComboBox.SetSelectedItem(selectedRow);
    
    disableEvents = false;              
}

function ProcessSettings(bool saving)
{
    local int i;
    local string s;
    
    if (! saving)
    {
        Settings.SetDefaults();
        SetMutators();        
    }
    else
    {
        Settings.Mutators = "";
        
        for (i = 0; i < MutatorNameList.Length; i++)
        {
		        s = mutatorComboBox.ComboList.GetElementValue(i);
            if (mid(s, 0, 1) == ">")
            {
                if (Settings.Mutators != "")
                {
                    Settings.Mutators = Settings.Mutators $ ",";
                }
                
                Settings.Mutators = Settings.Mutators $ MutatorClassList[i];
            }
        }
    }
    
    ProcessValue("MaxMultiJump", Settings.MaxMultiJump, saving);
    ProcessValue("BallSpeed", Settings.BallSpeed, saving);
    ProcessValue("DisplayMiniMap", Settings.DisplayMiniMap, saving);
    ProcessValue("WorldGravityZ", Settings.WorldGravityZ, saving);
    ProcessValue("MultiJumpBoost", Settings.MultiJumpBoost, saving);
    ProcessValue("AllowDodgeJumping", Settings.AllowDodgeJumping, saving);
    ProcessValue("NecrisTheme", Settings.NecrisTheme, saving);
    ProcessValue("DisableShootingScore", Settings.DisableShootingScore, saving);
    ProcessValue("StandardUTFeel", Settings.StandardUTFeel, saving);
    ProcessValue("GameSpeedMultiplier", Settings.GameSpeedMultiplier, saving);
    ProcessValue("AllowTranslocators", Settings.AllowTranslocators, saving);
    ProcessValue("AllowHoverboards", Settings.AllowHoverboards, saving);
    ProcessValue("AllowVehicles", Settings.AllowVehicles, saving);
    ProcessValue("AllowBallCarrierWeapons", Settings.AllowBallCarrierWeapons, saving);
    ProcessValue("ShowEnemyTeamOnMiniMap", Settings.ShowEnemyTeamOnMiniMap, saving);
    ProcessValue("ShowHomeTeamOnMiniMap", Settings.ShowHomeTeamOnMiniMap, saving);
    ProcessValue("CarryBallInVehicles", Settings.CarryBallInVehicles, saving);
    ProcessValue("BallCarrierHealthRegenerated", Settings.BallCarrierHealthRegenerated, saving); 
}

function bool OnRawInputKey (const out InputEventParameters EventParms)
{
  if ((EventParms.InputKeyName=='Escape') && EventParms.EventType == IE_Pressed) 
  {
    CloseScene(self);
  }
  return false;
}

function MarkCombo(string operation, UTUIComboBox combo)
{
    local int i;
    local string value;
    local name storeName;
    
    if (combo == mapComboBox)
    {
        storeName = 'BR_MAP_COMBO';
    }
    
    if (combo == mutatorComboBox)
    {
        storeName = 'BR_MUTATOR_COMBO'; 
    }
    
		value = combo.ComboList.GetElementValue(combo.ComboList.Index);

    if (mid(value, 0, 1) == ">")
    {
        if (operation == "MARK")
        {
            return;
        }
        
        value = mid(value, 1);
    }
    else
    {
       if (operation == "UNMARK")
       {
           return;
       }
       
       value = ">" $ value;
    }

    disableEvents = true;  
                
    i = combo.ComboList.Index;    
    StringStore.RemoveStrByIndex(storeName, i);
    StringStore.InsertStr(storeName, value, i);
       
    combo.ComboEditbox.SetValue(value);
    combo.SetSelectedItem(i);
        
    disableEvents = false;   
}

function string CurrentMap()
{
    local string map;
    
    map = mapComboBox.ComboList.GetElementValue(mapComboBox.ComboList.Index);
    if (map == "None")
    {
        map = "";
    }
    
    if (mid(map, 0, 1) == ">")
    {
        map = mid(map, 1);
    }    
    
    return map;  
}

function bool OnClicked(UIScreenObject EventObject, int PlayerIndex)
{   
    if (EventObject==Button_Save)
    {
        ProcessSettings(true);

        PlayerManager.SaveSettings(CurrentMap(), Settings.Serialize(), Settings.Mutators);
        
        if (mapComboBox.ComboList.Index > 0)
        {
             MarkCombo("MARK", mapComboBox);                         
        }
        else
        {
             CloseScene(self);         
        }        
    }
    
    if (EventObject==Button_Cancel)
    {
        CloseScene(self);       
    }
    
    if (EventObject == Button_Delete)
    {         
        PlayerManager.DeleteSettings(CurrentMap());

        Settings.ResetToDefaults();
        ProcessSettings(false);
        
        if (mapComboBox.ComboList.Index > 0)
        {
             MarkCombo("UNMARK", mapComboBox);
        }
        else
        {
            CloseScene(self);
        }
    }
    
    if (EventObject == Button_Defaults)
    {
        Settings.ResetToDefaults();
        ProcessSettings(false);
    }
    
    if (EventObject == Button_Restart)
    {
        PlayerManager.RestartGame();
        CloseScene(self); 
    }    
    	
      
    return true;
}

function bool CloseScene(UIScene SceneToClose, bool bSkipKismetNotify=false, bool bSkipAnimation=false)
{
    //events still get called after closing
    DisableEvents = true;
    
    return super.CloseScene(SceneToClose, bSkipKismetNotify, bSkipAnimation);
}

function bool OnQueryToolTip(UIObject Sender, out UIToolTip CustomToolTip )
{
   local string s1, s2;
   local int i;
   
   //UIObject.TOOLTIP_BINDING_INDEX = 100
   s1 = Sender.GetDefaultDataBinding(100);
   s2 = "";
   i = instr(s1, "|");
   if (i > 0)
   {
       s2 = mid(s1, i + 1);
       s1 = mid(s1, 0, i);
   }

   Label_ToolTip.SetValue(s1);
   Label_ToolTip2.SetValue(s2);
            
   return true; 
}

function  OnPageActivated( UITabControl Sender, UITabPage NewlyActivePage, int PlayerIndex )
{
    Label_ToolTip.SetValue("");
    Label_ToolTip2.SetValue("");    
}

defaultproperties
{
  bPauseGameWhileActive=False
  __OnRawInputKey__Delegate=Default__UTBRGameSettingsScene.OnRawInputKey  
  Name="Default__UTBRGameSettingsScene"
}
